/*
 * Decompiled with CFR 0.152.
 */
package gov.cms.oce.ext.table.util;

import gov.cms.oce.ext.table.util.Combination;
import gov.cms.oce.ext.table.util.CombinationOptimizer;
import gov.cms.oce.ext.table.util.CombinationOptimizerEquals;
import gov.cms.oce.ext.table.util.CombinationOptimizerIntersect;
import gov.cms.oce.ext.table.util.CombinationOptimizerRepeated;
import gov.cms.oce.ext.table.util.CombinationOptimizerThreeStep;
import gov.cms.oce.ext.table.util.OptimizerMode;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.StringJoiner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NcciOptimizationMain {
    private static final Logger logger = LoggerFactory.getLogger(NcciOptimizationMain.class);

    public static void main(String[] args) throws IOException {
        CommandLineParams params = new CommandLineParams(args);
        File ncciTableFile = new File(params.tableInputFile);
        CombinationOptimizer.LinkedHashMultimap<String, String> ncciTableData = NcciOptimizationMain.loadNcciTable(ncciTableFile);
        logger.info("Optimizing NCCI table with {}...", (Object)params);
        CombinationOptimizer<String> optimizer = NcciOptimizationMain.newOptimizer(params);
        List<Combination<String>> optimized = optimizer.optimize(ncciTableData);
        logger.info("Total groups (max group ID): " + optimized.size());
        File ncciFileLeftCodes = new File(params.leftCodesOutputFile);
        File ncciFileRightCodes = new File(params.rightCodesOutputFile);
        NcciOptimizationMain.writeOutComboFiles(optimized, ncciFileLeftCodes, ncciFileRightCodes);
        File ncciFileExpanded = new File(params.expandedCodesOutputFile);
        NcciOptimizationMain.writeOutExpandedFiles(optimized, ncciFileExpanded);
        logger.info("Done");
    }

    private static CombinationOptimizer<String> newOptimizer(CommandLineParams params) {
        switch (params.optimizerMode) {
            case EQUALS_MATCH: {
                return new CombinationOptimizerEquals();
            }
            case INTERSECT_SEARCH: {
                return new CombinationOptimizerIntersect(params.intersectionResultsLookup);
            }
            case THREE_STEP: {
                return new CombinationOptimizerThreeStep(params.intersectionResultsLookup);
            }
            case REPEATED: {
                return new CombinationOptimizerRepeated(params.repeatedMaxIterations, params.intersectionResultsLookup);
            }
        }
        throw new RuntimeException("Mode " + (Object)((Object)params.optimizerMode) + " not implemented");
    }

    public static CombinationOptimizer.LinkedHashMultimap<String, String> loadNcciTable(File ncciTableFile) throws IOException {
        List<String> lines = Files.readAllLines(ncciTableFile.toPath());
        lines = lines.subList(1, lines.size());
        CombinationOptimizer.LinkedHashMultimap<String, String> allCodes = new CombinationOptimizer.LinkedHashMultimap<String, String>();
        for (String line : lines) {
            String[] columns = NcciOptimizationMain.parseSimpleCsv(line);
            String leftCode = columns[0];
            String rightCode = columns[1];
            String lowRange = columns[2];
            String highRange = columns[3];
            String modIndicator = columns[4];
            String key = NcciOptimizationMain.createTableKey(leftCode, lowRange, highRange, modIndicator);
            allCodes.putSingle(key, rightCode);
        }
        logger.info("Loaded from NCCI table input file: {}", (Object)ncciTableFile);
        logger.info("  Total entries: {}", (Object)lines.size());
        logger.info("  Total unique left codes (uniqueness includes low/high/mod): {}", (Object)allCodes.keySet().size());
        logger.info("  Total unique right codes: {}", (Object)new HashSet(allCodes.values()).size());
        return allCodes;
    }

    public static void writeOutComboFiles(Collection<Combination<String>> combos, File leftCodesFile, File rightCodesFile) throws IOException {
        logger.info("Writing left codes and right code groups separate:");
        logger.info("Left codes file: {}", (Object)leftCodesFile);
        logger.info("Right codes file: {}", (Object)rightCodesFile);
        LinkedHashSet<String> leftCodes = new LinkedHashSet<String>();
        LinkedHashSet<String> rightCodes = new LinkedHashSet<String>();
        leftCodes.add(NcciOptimizationMain.formatSimleCsv("CODE", "LO_VERSION", "HI_VERSION", "MOD_INDICATOR", "GROUP_ID"));
        rightCodes.add(NcciOptimizationMain.formatSimleCsv("GROUP_ID", "CODE"));
        int groupId = 1;
        for (Combination<String> combo : combos) {
            for (String left : combo.getLeft()) {
                String[] leftData = NcciOptimizationMain.splitTableKey(left);
                leftCodes.add(NcciOptimizationMain.formatSimleCsv(leftData[0], leftData[1], leftData[2], leftData[3], groupId));
            }
            for (String rightCode : combo.getRight()) {
                rightCodes.add(NcciOptimizationMain.formatSimleCsv(groupId, rightCode));
            }
            ++groupId;
        }
        Files.deleteIfExists(leftCodesFile.toPath());
        Files.createDirectories(leftCodesFile.toPath().getParent(), new FileAttribute[0]);
        Files.write(leftCodesFile.toPath(), leftCodes, StandardOpenOption.CREATE);
        Files.deleteIfExists(rightCodesFile.toPath());
        Files.createDirectories(rightCodesFile.toPath().getParent(), new FileAttribute[0]);
        Files.write(rightCodesFile.toPath(), rightCodes, StandardOpenOption.CREATE);
    }

    public static void writeOutExpandedFiles(Collection<Combination<String>> combos, File expandedCodesFile) throws IOException {
        logger.info("Writing expanded codes file: {}", (Object)expandedCodesFile);
        ArrayList<String> expandedCodes = new ArrayList<String>();
        for (Combination<String> combo : combos) {
            for (String left : combo.getLeft()) {
                String[] leftData = NcciOptimizationMain.splitTableKey(left);
                for (String right : combo.getRight()) {
                    expandedCodes.add(NcciOptimizationMain.formatSimleCsv(leftData[0], right, leftData[1], leftData[2], leftData[3]));
                }
            }
        }
        Collections.sort(expandedCodes);
        expandedCodes.add(0, NcciOptimizationMain.formatSimleCsv("CODE1", "CODE2", "LO_VERSION", "HI_VERSION", "MOD_INDICATOR"));
        Files.deleteIfExists(expandedCodesFile.toPath());
        Files.createDirectories(expandedCodesFile.toPath().getParent(), new FileAttribute[0]);
        Files.write(expandedCodesFile.toPath(), expandedCodes, StandardOpenOption.CREATE);
    }

    private static String[] parseSimpleCsv(String line) {
        String[] columns = line.split(",");
        String[] result = new String[columns.length];
        int index = 0;
        for (String col : columns) {
            if (col.startsWith("\"")) {
                col = col.substring(1);
            }
            if (col.endsWith("\"")) {
                col = col.substring(0, col.length() - 1);
            }
            result[index] = col;
            ++index;
        }
        return result;
    }

    public static String formatSimleCsv(Object ... values) {
        StringJoiner sj = new StringJoiner("\",\"", "\"", "\"");
        for (Object val : values) {
            sj.add(String.valueOf(val));
        }
        return sj.toString();
    }

    private static String createTableKey(String leftCode, String lowRange, String highRange, String modIndicator) {
        return leftCode + "-" + lowRange + "-" + highRange + "-" + modIndicator;
    }

    public static String[] splitTableKey(String key) {
        String[] leftData = key.split("-");
        String leftCode = leftData[0];
        String lowRange = leftData[1];
        String highRange = leftData[2];
        String modIndicator = leftData[3];
        return new String[]{leftCode, lowRange, highRange, modIndicator};
    }

    private static final class CommandLineParams {
        public final String tableInputFile;
        public final String leftCodesOutputFile;
        public final String rightCodesOutputFile;
        public final String expandedCodesOutputFile;
        public final boolean intersectionResultsLookup;
        public final Integer repeatedMaxIterations;
        public final OptimizerMode optimizerMode;

        public CommandLineParams(String[] args) {
            if (args.length < 3) {
                throw new RuntimeException("First 3 parameters are mandatory");
            }
            this.tableInputFile = args[0];
            this.leftCodesOutputFile = args[1];
            this.rightCodesOutputFile = args[2];
            if (args.length > 3) {
                String optimizerModeParamValue = args[3];
                this.optimizerMode = OptimizerMode.valueOf(optimizerModeParamValue);
            } else {
                this.optimizerMode = OptimizerMode.INTERSECT_SEARCH;
            }
            this.expandedCodesOutputFile = args.length > 4 ? args[4] : null;
            this.intersectionResultsLookup = args.length > 5 ? Boolean.parseBoolean(args[5]) : false;
            this.repeatedMaxIterations = args.length > 6 && !args[6].equalsIgnoreCase("null") ? Integer.valueOf(Integer.parseInt(args[6])) : null;
        }

        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append(OptimizerMode.class.getSimpleName());
            sb.append(": ");
            sb.append((Object)this.optimizerMode);
            return sb.toString();
        }
    }
}

